//
//	GSSerialMacOS.cpp
//	 1998, 1999, 2000 Kyle Hammond
//	hammo009@tc.umn.edu
//	Use at your own risk.  All rights reserved.  Do not distribute without permission.
//

#ifndef __COMMRESOURCES__
	#include <CommResources.h>
#endif
#ifndef __CRMSERIALDEVICES__
	#include <CRMSerialDevices.h>
#endif
#if !defined( __PLSTRINGFUNCS__ )
	#include <PLStringFuncs.h>
#endif
#ifndef __SERIAL__
	#include <Serial.h>
#endif
#ifndef __OPENTPTCLIENT__
	#include <OpenTptClient.h>
#endif
#if !defined( __OPENTPTCONFIG__ )
	#include <OpenTptConfig.h>
#endif
#ifndef __OPENTPTSERIAL__
	#include <OpenTptSerial.h>
#endif
#ifndef __OPENTPTLINKS__
	#include <OpenTptLinks.h>
#endif

#include "GSSerialMacOS.h"

// NOTE: the outOptions.buf must be deleted by the caller of this function.
// Use delete [ ] outOptions->buf;
static OSStatus local_createOTSerialOptions( const UInt32 inBaudRate, const UInt32 inDataBits, const UInt32 inStopBits,
			const UInt32 inParity, TNetbuf *outOptions );

#pragma mark -

pascal void GSSerialOT::_class_serialEventHandler( void *contextPtr, OTEventCode code, OTResult result, void *cookie )
{
#pragma unused( cookie )
	register SerialStateInfoOT *info = &reinterpret_cast<GSSerialOT *>( contextPtr )->mSerialInfoOT;

	switch (code) {
		// We could poll for the data in our simple example here, but if we were doing
		// more complex things than just scanning the keyboard for keys, that would
		// be inconvient.  By using DTInstall, we can read our characters in a deferred
		// task right after the next interrupt time, and then output them however we want to.
	case T_DATA:
		// We have incoming data.
		if ( !info->deferredTaskScheduled ) {
			// If the deferred task is not scheduled right now, install it.
			info->error = DTInstall( &info->receiveDataDeferredTask );
			info->deferredTaskScheduled = true;
		}
	break;

	case T_GODATA:
	break;

		// Our Bind is completing - if it succeeded, issue a Connect() if we are supposed
		// to be connecting.  Otherwise, just sit tight until we get a T_LISTEN event that
		// indicates someone is trying to talk to us on the serial port.
	case T_BINDCOMPLETE:
		if ( result != kOTNoError )
			info->error = (OSStatus)result;
		else
			info->unbound = false;

		if ( info->error == kOTNoError && info->connect ) {
			TCall			req;

			req.addr.buf	= nil;
			req.addr.len	= 0;
			req.opt.buf	= nil;
			req.opt.len	= 0;
			req.udata.buf	= nil;
			req.udata.len	= 0;

			info->error = OTConnect(info->endPt, &req, nil);
#if !defined( TARGET_API_MAC_CARBON ) || ( TARGET_API_MAC_CARBON == 0 )
			if (info->error == kEBUSYErr) {
				OTClientList	clientList;

				// I don't know how to get the Provider Port Ref under Carbon.
				info->error = OTYieldPortRequest( info->endPt, OTGetProviderPortRef( info->endPt ), &clientList, sizeof(clientList) );
				if (info->error == noErr)
					info->error = OTConnect(info->endPt, &req, nil);
			}
#endif
		}

		if ( info->error != kOTNoError )
			info->ready = true;
	break;

		// Unbind is complete
	case T_UNBINDCOMPLETE:
//		if ( result != 0 )
//			DebugStr("\pAsync Unbind failed.");
		info->error	= (OSErr)result;
		info->unbound	= true;
	break;

		// An Option Management call is complete
	case T_OPTMGMTCOMPLETE:
//		if ( result != 0 )
//			DebugStr("\pOption Management call failed.");
		if ( info->error == kOTNoError)
			info->error = (OSStatus)result;
		info->optMgmtDone	= true;
	break;

		// a SndDisconnect has completed.  If info->error is not kOTNoError, then we
		// did the disconnect because of an error, so let's not overwrite the error.
		// Then, let's do an unbind.
	case T_DISCONNECTCOMPLETE:
//		if ( result != 0 )
//			DebugStr("\pAsync disconnect failed.");
		if ( info->error == kOTNoError )
			info->error = (OSStatus)result;
		info->disconnected	= true;
		OTUnbind(info->endPt);
	break;

		// Our connect request is complete, do a RcvConnect if all is well.
	case T_CONNECT:
//		if ( result != 0 )
//			DebugStr("\pAsync connect failed.");
		info->error = (OSErr)result;

		if ( info->error == kOTNoError ) {
			TCall retCall;
				
			retCall.addr.buf		= nil;
			retCall.addr.maxlen	= 0;
			retCall.opt.buf		= nil;
			retCall.opt.maxlen	= 0;
			retCall.udata.buf	= nil;
			retCall.udata.maxlen	= 0;
			
			info->error = OTRcvConnect(info->endPt, &retCall);
		}
		info->ready = true;
	break;

		// Our Accept is complete - we're ready for action!
	case T_ACCEPTCOMPLETE:
		info->acceptDone = true;
		if ( result != kOTNoError ) {
//			DebugStr("\pAsync accept failed.");
			info->error = (OSErr)result;
			info->ready = true;
		}
		else if ( info->passconDone ) {
			info->error = kOTNoError;
			info->ready = true;
		}
	break;

		// Our endpoint is now ready for action - we've got a connection
	case T_PASSCON:
		// T_PASSCON's can't have an error
		info->passconDone = true;
		if ( info->acceptDone ) {
			info->error = kOTNoError;
			info->ready = true;
		}
	break;

		// We have an incoming connection request.  We do a Listen to get the information
		// on the request (for Serial connections, this is really a formality, but...)
		// then issue an Accept() on the same endpoint.
		// If something goes wrong, we issue a SndDisconnect(), which has the effect of
		// rejecting the connection request.
	case T_LISTEN:
	{	TCall		lCall;

		lCall.addr.buf		= nil;
		lCall.addr.maxlen	= 0;
		lCall.opt.buf		= nil;
		lCall.opt.maxlen	= 0;
		lCall.udata.buf		= nil;
		lCall.udata.maxlen	= 0;

		info->error = OTListen(info->endPt, &lCall);
		if ( info->error != kOTNoError )
			info->ready = true;
		else {
			TCall*	aCall = new TCall;
			if ( aCall == nil ) {
				info->error = kOTOutOfMemoryErr;
				info->ready = true;
				OTSndDisconnect(info->endPt, nil);
				break;
			}
			aCall->addr.buf		= nil;	// Accept doesn't look at or use the "addr" field
			aCall->addr.maxlen	= 0;
			aCall->opt.buf		= nil;
			aCall->opt.len		= 0;
			aCall->udata.buf	= nil;
			aCall->udata.len	= 0;
			aCall->sequence	= lCall.sequence;
			info->error = OTAccept(info->endPt, info->endPt, aCall);
			if ( info->error != kOTNoError ) {
				delete aCall;
				info->ready = true;
				OTSndDisconnect(info->endPt, nil);
			}
		}
	}
	break;
		
	default:
//		DebugStr("\pUnknown or unexpected event");
	break;
	}
} // GSSerialOT::class_serialEventHandler


#if TARGET_CPU_68K
	// Deferred tasks on 68K use register based parameters, so we need to use some assembly glue.

inline asm long GetA1( void )
{	MOVE.L A1,(SP)
}

pascal void GSSerialOT::_class_deferredTaskProc68K( void )
{	long		myParm;

	myParm = GetA1( );					// retrieve parameter put in register A1
	_class_deferredTaskProc( myParm );	// run the deferred task
}

#endif

pascal void GSSerialOT::_class_deferredTaskProc( long dtParm )
{	SerialStateInfoOT	*info = &reinterpret_cast<GSSerialOT *>( dtParm )->mSerialInfoOT;
	OTResult			result, index;
	UInt8			buf[ kGSSerialMacBufferSize ];
	OTFlags			flags;

	while ( true ) {
		// Keep receiving until there is no data.
		result = OTRcv( info->endPt, buf, kGSSerialMacBufferSize, &flags );
		if ( result < 0 )
//			if ( result != kOTNoDataErr ) DebugStr("\pError in Rcv");
			break;
		else {
			// Place the data into the buffer at the end.
			// Note that excess characters are simply dropped.
			// This is probably OK for Videodisc players, since they don't usually send back much data.
			for ( index = 0; index < result && info->charsInBuffer < kGSSerialMacBufferSize; ++index )
				info->inBuffer[info->charsInBuffer++] = buf[index];
		}
	}

	info->deferredTaskScheduled = false;
	info->scheduled = false;
}

unsigned long	GSSerialOT::InitSerialComm(void)
{	OSStatus		errStatus;
	TBind		bind;
	Str63		tempString;

	// Initialize OpenTransport.
	errStatus = InitOpenTransport( );
	if (errStatus != kOTNoError)
		return errStatus;

	//	Set up the DeferredTask structure and allocate the UPP.
#if TARGET_CPU_68K
	mDeferredTaskUPP = NewDeferredTaskUPP( _class_deferredTaskProc68K );
#else
	mDeferredTaskUPP = NewDeferredTaskUPP( _class_deferredTaskProc );
#endif

	mSerialInfoOT.receiveDataDeferredTask.qLink = nil;
	mSerialInfoOT.receiveDataDeferredTask.qType = dtQType;
	mSerialInfoOT.receiveDataDeferredTask.dtFlags = 0;
	mSerialInfoOT.receiveDataDeferredTask.dtAddr = mDeferredTaskUPP;
	mSerialInfoOT.receiveDataDeferredTask.dtParam = reinterpret_cast<long>( this );
	mSerialInfoOT.receiveDataDeferredTask.dtReserved = 0;

	mSerialInfoOT.deferredTaskScheduled = false;

	mSerialInfoOT.unbound		= true;
	mSerialInfoOT.charsInBuffer	= 0;	// The input buffer is currently empty.

	// Create the Serial Port endpoint.
	PLstrcpy( tempString, mCurrentPortName.inPortName );
#if !defined( TARGET_API_MAC_CARBON ) || ( TARGET_API_MAC_CARBON == 0 )
	p2cstr( tempString );
#else
	CopyPascalStringToC( tempString, (char *)tempString );
#endif
	mSerialInfoOT.endPt = OTOpenEndpoint( OTCreateConfiguration( reinterpret_cast<char *>( tempString ) ), 0, &mInfo,
				&errStatus );

	if (errStatus != kOTNoError)
		return errStatus;

	// Install notifier we're going to use for testing. Pass the StateInfo
	// structure as the 'context' value that the event handler will get back.
	// (Note, can not use zero as the context value)
	mNotificationUPP = NewOTNotifyUPP( _class_serialEventHandler );
	errStatus = OTInstallNotifier( mSerialInfoOT.endPt, mNotificationUPP, this );
	if (errStatus != kOTNoError)
		return errStatus;

	OTSetAsynchronous(mSerialInfoOT.endPt);		// Ensure endpoint is async
	OTSetNonBlocking(mSerialInfoOT.endPt);		// Ensure endpoint is non-blocking

	// We'll start off the bind, and let the asynchronous event handler take it from there.
	mSerialInfoOT.acceptDone		= false;
	mSerialInfoOT.passconDone	= false;
	mSerialInfoOT.ready			= false;
	mSerialInfoOT.error			= kOTNoError;
	mSerialInfoOT.connect		= true;
				
	bind.addr.buf	= nil;
	bind.addr.len	= 0;
	bind.qlen		= mSerialInfoOT.connect ? 0 : 1;

	errStatus = OTBind(mSerialInfoOT.endPt, &bind, nil);

	// Setup the port baud rate, etc. appropriately.
	if (errStatus == kOTNoError)
		errStatus = this->SetPortOptions( &mCurrentPortInfo, &mCurrentPortName );

	return errStatus;
}

unsigned long	GSSerialOT::SetPortOptions( const GSSerialPortInfoRecord *inPortInfo, const GSXSerialPortNameRecord *inPortName )
{	TOptMgmt		cmd;
	UInt32		stopBits, parity;
	OSStatus		errStatus;

	// Test if the port should be switched.
	if ( !PortRecordsEqual( inPortName, &mCurrentPortName ) ) {
		// The endpoint is setup on a different port than the one wanted, so...switch ports.
		// The only way I can see to do this is to close the endpoint and then open a new endpoint, so...
		errStatus = this->CloseSerialComm( );
		if ( errStatus == kOTNoError ) {
			mCurrentPortInfo = *inPortInfo;
			mCurrentPortName = *inPortName;
			errStatus = this->InitSerialComm( );
		}

		// Don't need to continue with this, since InitSerialComm sets up the port options.
		return errStatus;
	}

	mCurrentPortInfo = *inPortInfo;

	// Change my GSSerialPortInfoRecord structure into something Open Transport will understand.
	switch (inPortInfo->parity) {
	case kGSSerialParityOdd:		parity = kOTSerialOddParity;		break;
	case kGSSerialParityEven:	parity = kOTSerialEvenParity;		break;
	case kGSSerialParityNone:
	default:					parity = kOTSerialNoParity;		break;
	}

	switch (inPortInfo->stopBits) {
	case 	kGSSerialStopBits15:	stopBits = 15;	break;
	case kGSSerialStopBits20:	stopBits = 20;	break;
	case kGSSerialStopBits10:
	default:					stopBits = 10;	break;
	}

	cmd.opt.buf = nil;
	cmd.flags = T_NEGOTIATE;
	errStatus = local_createOTSerialOptions( inPortInfo->baud, inPortInfo->dataBits, stopBits, parity, &cmd.opt );

	if ( errStatus != kOTNoError )
		return errStatus;

	// Make the option management call to set stuff up
	mSerialInfoOT.optMgmtDone = false;
	errStatus = OTOptionManagement( mSerialInfoOT.endPt, &cmd, &cmd );

	if ( errStatus == kOTNoError ) {
		// Wait until the options have been changed.
		while ( !mSerialInfoOT.optMgmtDone ){
			unsigned long	finalTicks;
			Delay( 1, &finalTicks );
		}

		// See if there was an error setting the options.
		errStatus = mSerialInfoOT.error;
	}

	if ( cmd.opt.buf != nil )
	 	delete [ ] cmd.opt.buf;

	mCurrentPortInfo = *inPortInfo;

	return errStatus;
}

unsigned long	GSSerialOT::CloseSerialComm(void)
{	OSStatus	errStatus;

	// Clean up the endpoint we allocated
	if ( mSerialInfoOT.endPt != nil ) {
		OTResult state = OTGetEndpointState( mSerialInfoOT.endPt );
		
		if ( state == T_DATAXFER || state == T_INCON || state == T_OUTCON )
			OTSndDisconnect( mSerialInfoOT.endPt, nil );
		else if ( state == T_IDLE )
			OTUnbind( mSerialInfoOT.endPt );

		while ( !mSerialInfoOT.unbound ) {
			unsigned long	finalTicks;
			Delay( 1, &finalTicks );
		}

		OTRemoveNotifier( mSerialInfoOT.endPt );
		DisposeOTNotifyUPP( mNotificationUPP );
		mNotificationUPP = nil;
		OTSetSynchronous( mSerialInfoOT.endPt );
		OTSetBlocking( mSerialInfoOT.endPt );
		errStatus = OTCloseProvider( mSerialInfoOT.endPt );
		if ( errStatus != kOTNoError )
			return errStatus;
	}

	if ( !mSerialInfoOT.deferredTaskScheduled )
		DisposeDeferredTaskUPP( mDeferredTaskUPP );

	// Close OpenTransport.
	CloseOpenTransport( );

	return errStatus;
}

unsigned long	GSSerialOT::SendSerialInfo( const void *inBuffer, const unsigned long inBufferSize )
{	OSStatus		errStatus;
	OTResult		sent;
	char*		toSend = const_cast<char *>( inBuffer );
	unsigned long	bytesLeft = inBufferSize;

	// We're sending data, and we want a reply.
	mSerialInfoOT.scheduled = true;

	while ( true ) {
		// Try to send some data.
		sent = OTSnd( mSerialInfoOT.endPt, toSend, bytesLeft, T_MORE );

		// If we didn't send any or there was some error other than a Flow error, bail out.
		if ( sent < 0 && sent != kOTFlowErr ) {
			errStatus = sent;
			break;
		}

		// If we sent it all, we're done!
		if ( sent == bytesLeft ) {
			errStatus = noErr;
			break;
		}

		// If we sent some, but not all:
		// attempt to send the rest of the data.
		if ( sent > 0 ) {
			toSend		+= sent;
			bytesLeft		-= (OTByteCount)sent;
		}
	}

	return errStatus;
}

// Look for up to (count) characters, and return how many characters there are.
// Returns either kOTNoDataErr or noErr.
unsigned long	GSSerialOT::GetSerialInfo( void *outBuffer, unsigned long *ioCount, const long inWaitTicks )
{	UInt8		index;
	unsigned long	finishTime, finalTicks;

	if ( mSerialInfoOT.scheduled || mSerialInfoOT.charsInBuffer < *ioCount )
		Delay( 1, &finalTicks );

	// If we're still scheduled to have data coming, or there isn't enough data yet...
	if ( mSerialInfoOT.scheduled || mSerialInfoOT.charsInBuffer < *ioCount ) {
		finishTime = TickCount( ) + inWaitTicks;

		// Wait for the specified amount of time, or until there's enough characters.
		while ( TickCount( ) < finishTime && mSerialInfoOT.charsInBuffer < *ioCount )
			Delay( 1, &finalTicks );
	}

	if ( mSerialInfoOT.charsInBuffer == 0 )
		return kOTNoDataErr;

	// Copy the characters out of the incoming buffer.
	for (index = 0; index < *ioCount && index < mSerialInfoOT.charsInBuffer; index++)
		reinterpret_cast<char *>( outBuffer )[index] = mSerialInfoOT.inBuffer[index];

	// This is how many characters we've copied.
	*ioCount = index;
	mSerialInfoOT.charsInBuffer -= index;

	// Now move the characters in the buffer forward that many characters.
	for ( index = 0; index < mSerialInfoOT.charsInBuffer; index++ )
		mSerialInfoOT.inBuffer[ index ] = mSerialInfoOT.inBuffer[ index + (*ioCount) ];

	return noErr;
}

short GSSerialOT::FindReadySerialPorts( GSXSerialPortNameRecordHandle *outPortInfo, const bool inOpenAndClose )
{	// This lists all unique serial ports.  The ready state determines whether the port is in use or not.
	OTPortRecord			record;
	register int			i;
	UInt16				deviceType;
	Boolean				ready;
	OSStatus				errStatus;
	OSErr				err;
	int					index, count = 0;

	if ( outPortInfo == nil )
		return 0;

	if ( inOpenAndClose ) {
#if !defined( TARGET_API_MAC_CARBON ) || ( TARGET_API_MAC_CARBON == 0 )
		// Initialize OpenTransport utilities only.  All we need to do is look at the ports installed.
		errStatus = InitOpenTransportUtilities( );
#else
		errStatus = InitOpenTransport( );
#endif
		if (errStatus != kOTNoError)
			return 0;
	}

	for (i = 0; OTGetIndexedPort(&record, i); i++) {
		deviceType = OTGetDeviceTypeFromPortRef( record.fRef );		
		// Check that it's a serial port and not an alias to another port.
		if (deviceType == kOTSerialDevice && !(record.fInfoFlags & kOTPortIsAlias))
			// Actually, I want to report the port even if it's in use.
//			ready = (record.fPortFlags == 0);
			ready = true;
		else
			ready = false;

		if (ready) {
			if ( *outPortInfo == nil )
				*outPortInfo = (GSXSerialPortNameRecordHandle)NewHandle( sizeof(GSXSerialPortNameRecord) );
			else
				SetHandleSize( (Handle)*outPortInfo, GetHandleSize( (Handle)*outPortInfo )
							+ sizeof(GSXSerialPortNameRecord) );
			err = MemError( );
			if ( err == noErr ) {
				count++;
				index = ( GetHandleSize( (Handle)*outPortInfo ) / sizeof(GSXSerialPortNameRecord) ) - 1;

				HLock( reinterpret_cast<Handle>( *outPortInfo ) );

#if !defined( TARGET_API_MAC_CARBON ) || ( TARGET_API_MAC_CARBON == 0 )
				// This is lame for the built in ports on 680x0 machines.
				// They all report: Serial Built-In
				OTGetUserPortNameFromPortRef( record.fRef, (**outPortInfo)[ index ].userPortName );

				c2pstr( record.fPortName );
				PLstrcpy( (**outPortInfo)[ index ].inPortName, (unsigned char *)record.fPortName );
				PLstrcpy( (**outPortInfo)[ index ].outPortName, (unsigned char *)record.fPortName );
				p2cstr( (unsigned char *)record.fPortName );
#else
				// This is even more lame under Carbon - No way to get user visible name???
				CopyCStringToPascal( record.fPortName, (**outPortInfo)[ index ].userPortName );

				CopyCStringToPascal( record.fPortName, (**outPortInfo)[ index ].inPortName );
				CopyCStringToPascal( record.fPortName, (**outPortInfo)[ index ].outPortName );
#endif

				HUnlock( reinterpret_cast<Handle>( *outPortInfo ) );
			}
		}
	}

	if ( inOpenAndClose )
		// Close OpenTransport.
		CloseOpenTransport( );

	return count;
}

// NOTE: the outOptions.buf must be deleted by the caller of this function.
// Use delete [ ] outOptions->buf;
static OSStatus local_createOTSerialOptions( const UInt32 inBaudRate, const UInt32 inDataBits, const UInt32 inStopBits,
			const UInt32 inParity, TNetbuf *outOptions )
{	TOption		*optPtr;
	unsigned char	*optionBuffer = nil;

#if __option( exceptions )
	try {
#endif
		optionBuffer = new unsigned char[ ( sizeof(TOption) * 4 ) ];
#if __option( exceptions )
	}
	catch ( ... ) {
		return memFullErr;
	}
#endif

	if ( optionBuffer == nil )
		return memFullErr;

	// Set the baud rate.
	optPtr = reinterpret_cast<TOption *>( &optionBuffer[ 0 ] );
	optPtr->len = sizeof(TOption);
	optPtr->level = COM_SERIAL;
	optPtr->status = 0;
	optPtr->name = SERIAL_OPT_BAUDRATE;
	optPtr->value[ 0 ] = inBaudRate;

	// Set the data bits.
	optPtr = reinterpret_cast<TOption *>( &optionBuffer[ sizeof(TOption) ] );
	optPtr->len = sizeof(TOption);
	optPtr->level = COM_SERIAL;
	optPtr->status = 0;
	optPtr->name = SERIAL_OPT_DATABITS;
	optPtr->value[ 0 ] = inDataBits;

	// Set the stop bits.
	optPtr = reinterpret_cast<TOption *>( &optionBuffer[ sizeof(TOption) * 2 ] );
	optPtr->len = sizeof(TOption);
	optPtr->level = COM_SERIAL;
	optPtr->status = 0;
	optPtr->name = SERIAL_OPT_STOPBITS;
	optPtr->value[ 0 ] = inStopBits;

	// Set the parity.
	optPtr = reinterpret_cast<TOption *>( &optionBuffer[ sizeof(TOption) * 3 ] );
	optPtr->len = sizeof(TOption);
	optPtr->level = COM_SERIAL;
	optPtr->status = 0;
	optPtr->name = SERIAL_OPT_PARITY;
	optPtr->value[ 0 ] = inParity;

	// Fill in the outOptions struct.
	outOptions->buf	 = optionBuffer;
	outOptions->maxlen = outOptions->len = sizeof(TOption) * 4;

	return noErr;
}

#pragma mark -

#if !defined( TARGET_API_MAC_CARBON ) || ( TARGET_API_MAC_CARBON == 0 )

// Don't need the rest of this file if we're compiling under Carbon.

class GSSerialMacAsyncHelper {
	// This class serves to keep the buffers around until the end of a PBReadAsync or PBWriteAsync call.
	// It is also used to implement a continous PBReadAsync call on the serial port; in the read completion routine,
	//   the launch function is called again.
public:
	GSSerialMacAsyncHelper( const short inDriverRefNum, const unsigned long inBufferSize,
				const bool inDeleteUponCompletion );

	// Call PBReadAsync if inSendBuffer == nil; otherwise call PBWriteAsync.
	OSErr launch( const void *inSendBuffer, GSSerialClassic *inComm );

	// finished( ) will return true when the asynchronous I/O is complete
	bool finished( void ) const {	return mFinished;	}

	// Only use these if finished is true.
	void *buffer( void ) const
	{	if ( mFinished )	return mBuffer;
		else			return nil;
	}
	unsigned long bufferSize( void ) const
	{	if ( mFinished )		return mData.paramBlock.ioParam.ioActCount;
		else				return 0;
	}
	OSErr completionError( void ) const {	return mData.paramBlock.ioParam.ioResult;	}

	// Destructor.
	~GSSerialMacAsyncHelper( );

// Implementation section.
private:
	struct AsyncSerialParamBlock {
		// Tack on a pointer to this object at the end of a ParamBlockRec.
		ParamBlockRec			paramBlock;
		GSSerialMacAsyncHelper	*serialObject;
	} mData;

	IOCompletionUPP	mReadCompletionUPP, mWriteCompletionUPP;
	OSErr			mConstructorError;
	bool				mFinished, mDeleteUponCompletion;
	Ptr				mBuffer;
	GSSerialClassic		*mComm;

	static pascal void _readComplete( ParmBlkPtr paramBlock );
	static pascal void _writeComplete( ParmBlkPtr paramBlock );
};

GSSerialMacAsyncHelper::GSSerialMacAsyncHelper( const short inDriverRefNum, const unsigned long inBufferSize,
				const bool inDeleteUponCompletion ) : mConstructorError( noErr ), mFinished( true ),
				mDeleteUponCompletion( inDeleteUponCompletion ), mComm( nil )
{	mReadCompletionUPP = NewIOCompletionProc( _readComplete );
	mWriteCompletionUPP = NewIOCompletionProc( _writeComplete );

	mBuffer = NewPtr( sizeof( inBufferSize ) );
	if ( mBuffer != nil ) {
		mData.paramBlock.ioParam.ioVRefNum	= 0;
		mData.paramBlock.ioParam.ioRefNum	= inDriverRefNum;
		mData.paramBlock.ioParam.ioBuffer		= mBuffer;
		mData.paramBlock.ioParam.ioReqCount	= inBufferSize;
		mData.paramBlock.ioParam.ioPosMode	= fsFromMark;
		mData.paramBlock.ioParam.ioPosOffset	= 0;

		mData.serialObject = this;
	} else
		mConstructorError = MemError( );
}

OSErr GSSerialMacAsyncHelper::launch( const void *inSendBuffer, GSSerialClassic *inComm )
{	OSErr	err;

	// Report the MemError( ) if the buffer could not be created.
	if ( mConstructorError != noErr )
		return mConstructorError;

	mComm = inComm;

	if ( inSendBuffer == nil  ) {
		mData.paramBlock.ioParam.ioCompletion = mReadCompletionUPP;
		err = PBReadAsync( reinterpret_cast<ParmBlkPtr>( &mData ) );
		if ( err == noErr )
			mFinished = false;
	} else {
		// Copy the outgoing data to the local buffer for writing.
		BlockMoveData( inSendBuffer, mBuffer, GetPtrSize( mBuffer ) );

		mData.paramBlock.ioParam.ioCompletion = mWriteCompletionUPP;
		err = PBWriteAsync( reinterpret_cast<ParmBlkPtr>( &mData ) );
		if ( err == noErr )
			mFinished = false;
	}

	return err;
}

GSSerialMacAsyncHelper::~GSSerialMacAsyncHelper( )
{	if ( mBuffer != nil )
		DisposePtr( mBuffer );
	DisposeRoutineDescriptor( mReadCompletionUPP );
	DisposeRoutineDescriptor( mWriteCompletionUPP );
}

pascal void GSSerialMacAsyncHelper::_readComplete( ParmBlkPtr inParamBlock )
{	GSSerialMacAsyncHelper	*object = reinterpret_cast<AsyncSerialParamBlock *>( inParamBlock )->serialObject;
	OSErr			err;
	register long		index;

	// We are done!
	object->mFinished = true;

	// Place the data into the buffer at the end.
	// Note that excess characters are simply dropped.
	// This is probably OK for Videodisc players, since they don't usually send back much data.
	for ( index = 0; index < inParamBlock->ioParam.ioActCount &&
					object->mComm->mCharsInBuffer < kGSSerialMacBufferSize; ++index )
		object->mComm->mIncomingBuffer[ object->mComm->mCharsInBuffer++ ] = object->mBuffer[ index ];

	if ( object->mComm->mContinuousRead )
		// Launch the GSSerialMacAsyncHelper object again.
		err = object->launch( nil, object->mComm );
	else {
		// Go ahead and delete the object, if it was a fire-and-forget read.
		if ( object->mDeleteUponCompletion )
			delete object;
	}
}

pascal void GSSerialMacAsyncHelper::_writeComplete( ParmBlkPtr inParamBlock )
{	GSSerialMacAsyncHelper	*object = reinterpret_cast<AsyncSerialParamBlock *>( inParamBlock )->serialObject;

	// We are done!
	object->mFinished = true;

	// Go ahead and delete the object, if it was a fire-and-forget write.
	if ( object->mDeleteUponCompletion )
		delete object;
}

#pragma mark -

// These are some routines taken from TechNote 1119 to use with classic serial ports.
static OSErr OpenOneSerialDriver( ConstStr255Param inDriverName, short *outRefNum );
static bool SerialArbitrationExists( void );
static bool DriverIsOpen( ConstStr255Param inDriverName );

static bool DriverIsOpen( ConstStr255Param inDriverName )
{	// Walks the unit table to determine whether the given driver is marked as open in the table.
	// Returns false if the driver is closed or does not exist.

	Boolean		found = false;
	bool			isOpen = false;
	short		unit = 0;
	DCtlHandle	dceHandle;
	StringPtr		namePtr;

	while ( !found && ( unit < LMGetUnitTableEntryCount( ) ) ) {
		// Get handle to a device control entry.
		// GetDCtlEntry takes a driver refNum but we convert between unit number and driver refNum
		// by using bitwise not.
		dceHandle = GetDCtlEntry( ~unit );

		if (dceHandle != nil && (**dceHandle).dCtlDriver != nil) {
			// If the driver is RAM based, dCtlDriver is a handle, otherwise it's a pointer.
			// We have to do some fancy casting to handle each case.
			if ( ((**dceHandle).dCtlFlags & dRAMBasedMask) != 0 )
				namePtr = & (**((DRVRHeaderHandle) (**dceHandle).dCtlDriver)).drvrName[0];
			else
				namePtr = & (*((DRVRHeaderPtr) (**dceHandle).dCtlDriver)).drvrName[0];

			// Now that we have a pointer to the driver name, compare it to the name we're looking for.
			if ( EqualString( inDriverName, namePtr, false, true ) ) {
				found = true;
				isOpen = ((**dceHandle).dCtlFlags & dOpenedMask) != 0;
			}
		}

		unit++;
	}

	return isOpen;
}

static bool SerialArbitrationExists( void )
{	// Test Gestalt to see if serial arbitration is installed on this machine.
	bool		result;
	long		response;

	enum {
		gestaltSerialPortArbitratorAttr = 'arb ',
		gestaltSerialPortArbitratorExists = 0
	};

	result = ( Gestalt(gestaltSerialPortArbitratorAttr, &response) == noErr &&
					( response & (1 << gestaltSerialPortArbitratorExists) != 0) != 0 );

	return result;
}

static OSErr OpenOneSerialDriver( ConstStr255Param inDriverName, short *outRefNum )
{	// The true way to open a serial driver.  From TechNote 1119.
	OSErr	err;

	if ( SerialArbitrationExists( ) )
		err = OpenDriver( inDriverName, outRefNum );
	else {
		if ( DriverIsOpen( inDriverName ) )
			err = portInUse;
		else
			err = OpenDriver( inDriverName, outRefNum );
	}

	return err;
}

unsigned long	GSSerialClassic::InitSerialComm( void )
{	OSErr		err;
	SerShk		flags = { 0, 0, 'X', 'x', 0, 0, 0, 0 };

	mContinuousRead = true;
	mCharsInBuffer = 0;

	err = OpenOneSerialDriver( mCurrentPortName.outPortName, &mOutRefNum );

	if (err == noErr)
		err = Control( mOutRefNum, kSERDHandshake, &flags);

	if (err == noErr)
		err = OpenOneSerialDriver( mCurrentPortName.inPortName, &mInRefNum );

	if (err == noErr)
		err = Control( mInRefNum, kSERDHandshake, &flags );

	if (err == noErr)
		err = SetPortOptions( &mCurrentPortInfo, &mCurrentPortName );

	if ( err == noErr && mAsync ) {
		// Set up for the asynchronous read.
		// Essentially, we're calling PBReadAsync all the time, waiting for data.
		// The GSSerialMacAsyncHelper object puts all the data into mIncomingBuffer.
		// Fire and forget about it (auto deleted in call to CloseSerialComm).
		GSSerialMacAsyncHelper	*helper = new GSSerialMacAsyncHelper( mInRefNum, 1, true );

		err = helper->launch( nil, this );

		if ( err != noErr )
			delete helper;
	}

	return err;
}

unsigned long	GSSerialClassic::SetPortOptions( const GSSerialPortInfoRecord *inPortInfo, const GSXSerialPortNameRecord *inPortName )
{	OSErr	err;
	short	serConfig = 0;

	if ( !PortRecordsEqual( inPortName, &mCurrentPortName ) ) {
		// Need to switch the port.
		err = CloseSerialComm( );
		if (err == noErr) {
			mCurrentPortInfo = *inPortInfo;
			mCurrentPortName = *inPortName;
			err = InitSerialComm( );
		}

		return err;
	}

	mCurrentPortInfo = *inPortInfo;

	// Convert the GSSerialPortInfoRecord options into classic serial driver configuration info.

	switch( inPortInfo->baud ) {
	case 150:		serConfig += baud150;	break;
	case 300:		serConfig += baud300;	break;
	case 600:		serConfig += baud600;	break;
	case 1200:	serConfig += baud1200;	break;
	case 1800:	serConfig += baud1800;	break;
	case 2400:	serConfig += baud2400;	break;
	case 3600:	serConfig += baud3600;	break;
	case 4800:	serConfig += baud4800;	break;
	case 7200:	serConfig += baud7200;	break;
	case 14400:	serConfig += baud14400;	break;
	case 19200:	serConfig += baud19200;	break;
	case 28800:	serConfig += baud28800;	break;
	case 38400:	serConfig += baud38400;	break;
	case 57600:	serConfig += baud57600;	break;
	case 9600:
	default:		serConfig += baud9600;	break;
	}

	switch ( inPortInfo->dataBits ) {
	case 5:	serConfig += data5;	break;
	case 6:	serConfig += data6;	break;
	case 7:	serConfig += data7;	break;
	case 8:
	default:	serConfig += data8;	break;
	}

	switch ( inPortInfo->stopBits ) {
	case kGSSerialStopBits15:	serConfig += stop15;	break;
	case kGSSerialStopBits20:	serConfig += stop20;	break;
	case kGSSerialStopBits10:
	default:					serConfig += stop10;	break;
	}

	switch ( inPortInfo->parity ) {
	case kGSSerialParityOdd:		serConfig += oddParity;	break;
	case kGSSerialParityEven:	serConfig += evenParity;	break;
	case kGSSerialParityNone:
	default:					serConfig += noParity;	break;
	}

	err = Control( mInRefNum, kSERDConfiguration, &serConfig );
	if (err == noErr)
		err = Control( mOutRefNum, kSERDConfiguration, &serConfig );

	mCurrentPortInfo = *inPortInfo;

	return err;
}

unsigned long	GSSerialClassic::CloseSerialComm(void)
{	OSErr	err;

	// Set this so any async read calls pending will not call PBReadAsync again.
	mContinuousRead = false;

	// Now KillIO on input and output buffers.  This will trigger the completion routines if we're async.
	(void)KillIO( mInRefNum );
	err = CloseDriver( mInRefNum );
	if ( err == noErr ) {
		(void)KillIO( mOutRefNum );
		err = CloseDriver( mOutRefNum );
	}

	return err;
}

unsigned long	GSSerialClassic::SendSerialInfo( const void *inBuffer, const unsigned long inBufferSize )
{	OSErr		err;

	// Send the command to the video disc player.
	if ( mAsync ) {
		// Fire and forget about it (auto deleted upon completion).
		GSSerialMacAsyncHelper	*helper = new GSSerialMacAsyncHelper( mOutRefNum, inBufferSize, true );

		err = helper->launch( inBuffer, this );
		if ( err != noErr )
			// Error!	Trash the helper.
			delete helper;
	} else {
		ParamBlockRec	paramBlock;

		paramBlock.ioParam.ioVRefNum		= 0;
		paramBlock.ioParam.ioRefNum		= mOutRefNum;
		paramBlock.ioParam.ioBuffer		= const_cast<char *>( inBuffer );
		paramBlock.ioParam.ioReqCount	= inBufferSize;
		paramBlock.ioParam.ioPosMode		= fsAtMark;
		paramBlock.ioParam.ioPosOffset	= 0;
		paramBlock.ioParam.ioCompletion	= nil;

		err = PBWriteSync( &paramBlock );
	}

	return err;
}

unsigned long GSSerialClassic::GetSerialInfo( void *outBuffer, unsigned long *ioCount, const long inWaitTicks )
{	OSErr	err = noErr;

	// Read back information from the serial port.
	if ( mAsync ) {
		UInt32	doneTime, index;

		if ( mCharsInBuffer < *ioCount )
			SystemTask( );

		// Need to hang around inWaitTicks, or until we have enough data.
		for ( doneTime = TickCount( ) + inWaitTicks; TickCount( ) < doneTime && mCharsInBuffer < *ioCount; )
			SystemTask( );

		// Should return a classic serial error here, but this fits pretty well.
		if ( mCharsInBuffer == 0 )
			err = kOTNoDataErr;

		if ( err == noErr ) {
			// Copy the characters out of the incoming buffer.
			for ( index = 0; index < *ioCount && index < mCharsInBuffer; index++ )
				reinterpret_cast<char *>( outBuffer )[ index ] = mIncomingBuffer[ index ];

			// This is how many characters we've copied.
			*ioCount = index;
			mCharsInBuffer -= index;

			// Now move the characters in the buffer forward that many characters.
			for ( index = 0; index < mCharsInBuffer; index++ )
				mIncomingBuffer[ index ] = mIncomingBuffer[ index + (*ioCount) ];
		}
	} else {
		// Perform a synchronous read.  This could block forever if no data is coming in.
		ParamBlockRec		paramBlock;

		paramBlock.ioParam.ioVRefNum		= 0;
		paramBlock.ioParam.ioRefNum		= mInRefNum;
		paramBlock.ioParam.ioReqCount	= *ioCount;
		paramBlock.ioParam.ioBuffer		= reinterpret_cast<char *>( outBuffer );
		paramBlock.ioParam.ioPosMode		= fsAtMark;
		paramBlock.ioParam.ioPosOffset	= 0;

		paramBlock.ioParam.ioCompletion = nil;
		err = PBReadSync( &paramBlock );
		*ioCount = paramBlock.ioParam.ioActCount;
	}

	return err;
}

short GSSerialClassic::FindReadySerialPorts( GSXSerialPortNameRecordHandle *outPortInfo, const bool inOpenAndClose )
{
#pragma unused( inOpenAndClose )
	CRMRec					commRecord;
	CRMRecPtr				thisCommRecord;
	CRMSerialPtr				serialPtr;
	OSErr					err;
	int						index, count = 0;

	if ( outPortInfo == nil )
		return 0;

	// Use the Communications Resource Manager (CRM) to determine what ports are available.
	// CRM is part of System 7.0+.

	(void)InitCRM( );

	// Set up commRecord to specify that we're interested in serial devices.
	commRecord.crmDeviceType = crmSerialDevice;
	commRecord.crmDeviceID = 0;

	// Repeatedly call CRMSearch to iterate through all serial ports.
	thisCommRecord = &commRecord;
	do {
		thisCommRecord = CRMSearch( thisCommRecord );
		if (thisCommRecord != nil) {
			// Cast crmAttributes to a CRMSerialPtr to access the serial specific fields about the port.
			serialPtr = (CRMSerialPtr) thisCommRecord->crmAttributes;

			if ( *outPortInfo == nil )
				*outPortInfo = (GSXSerialPortNameRecordHandle)NewHandle( sizeof(GSXSerialPortNameRecord) );
			else
				SetHandleSize( (Handle)*outPortInfo, GetHandleSize( (Handle)*outPortInfo )
							+ sizeof(GSXSerialPortNameRecord) );
			err = MemError( );
			if ( err == noErr ) {
				count++;
				index = ( GetHandleSize( (Handle)*outPortInfo ) / sizeof(GSXSerialPortNameRecord) ) - 1;
				PLstrcpy( (**outPortInfo)[ index ].userPortName, *serialPtr->name );
				PLstrcpy( (**outPortInfo)[ index ].inPortName, *serialPtr->inputDriverName );
				PLstrcpy( (**outPortInfo)[ index ].outPortName, *serialPtr->outputDriverName );
			}

			// Ensure that CRMSearch finds the next device.
			commRecord.crmDeviceID = thisCommRecord->crmDeviceID;
		}
	} while ( thisCommRecord != nil )	;

	return count;
}

#endif // ! TARGET_API_MAC_CARBON
